<?php
namespace Application\Controller;

use Application\Exception\EntryNotFoundException;
use Application\Model\UserTable;
use Application\Services\Mail\ChangedPasswordNotificationMailer;
use Application\Services\Mail\ForgotPasswordMailer;
use Laminas\Mvc\Controller\AbstractActionController;
use Laminas\View\Model\ViewModel;
use Application\Model\User;
use Laminas\View\Model\JsonModel;
use Laminas\Http\Client;
use Laminas\Http\Client\Adapter\Curl;

class LoginController extends AbstractActionController
{

    public function indexAction()
    {
        $authAdapter = $this->serviceLocator->get('AuthService');

        $redirect = urldecode($this->params()->fromQuery('redirect', '/wstr/application/members'));
        $redirect = urldecode($this->params()->fromPost('redirect', $redirect));

        $csrf_cookie = urldecode($this->params()->fromQuery('csrf_token', null));


        /**
         * Prevent cross site request forgeries
         */
        if ($csrf_cookie)
        {
            $session_csrf_cookie = $_SESSION['PN_CSRF_COOKIE'];
            if ($csrf_cookie !== $session_csrf_cookie)
            {
                $this->redirect()->toUrl('/wstr/application/login');
            }
        }

        /**
         * @var $userTable UserTable
         */
        $userTable = $this->serviceLocator->get('Application\Model\UserTable');


        if ($this->serviceLocator->get('AuthService')->hasIdentity())
        {
            /**
             * @var $user User
             */
            $user = $this->serviceLocator->get('ActiveUser');
            $user->lastlog = date("Y-m-d H:i:s");

            $userTable->saveOne($user);

            return $this->redirect()->toUrl($redirect);
        }

        $config = $this->serviceLocator->get('config');
        $captcha_key = $config['google_captcha']['site_key'];

        $this->layout('layout/logged-out');

        $viewModel = new ViewModel();
        $viewModel->setVariable('captcha_key', $captcha_key);
        $viewModel->setVariable('redirect', urlencode($redirect));

        if ($this->request->getMethod() == 'POST')
        {
            $captcha_secret = $config['google_captcha']['secret_key'];
            $captcha_api = $config['google_captcha']['api_url'];

            $captchaResponse = $this->request->getPost('g-recaptcha-response');

            $options = [
                'adapter' => Curl::class,
                'curloptions' => [
                    CURLOPT_FOLLOWLOCATION => true,
                    CURLOPT_POST => true,
                    // CURLOPT_USE_SSL => true,
                    // CURLOPT_CAINFO => '/etc/ssl/certs/ca-certificates.crt'
                ]
            ];
            new Client($captcha_api);
            $client = new Client($captcha_api, $options);
            $client->setParameterPost([
                'secret' => $captcha_secret,
                'response' => $captchaResponse,
                'remoteip' => $_SERVER['REMOTE_ADDR']
            ]);
            $client->setMethod('POST');
            $response = $client->send();

            $content = json_decode($response->getBody());

            if (!$content->success)
            {
                $submissionErrors = [];
                $submissionErrors[] = 'reCAPTCHA Anti-Robot test failed.  If you continue to experience problems please contact the help desk';
                $viewModel->setVariable('submissionErrors', $submissionErrors);
                $viewModel->setVariable('captcha_key', $captcha_key);
                $viewModel->setVariable('redirect', urlencode($redirect));
                return $viewModel;
            }

            $username = $this->request->getPost('username', null);
            $password = $this->request->getPost('password', null);

            if ($username == null || $password == null)
            {
                $submissionErrors = [];
                if ($username == null) $submissionErrors[] = 'Username not specified.';
                if ($password == null) $submissionErrors[] = 'Password not specified.';

                $viewModel->setVariable('submissionErrors', $submissionErrors);

                return $viewModel;
            }


            $storedUser = new User();
            $storedUser->username = $username;
            try
            {
                $storedUser = $userTable->findMatchingUser($storedUser);
            } catch (\Exception $e)
            {
                $storedUser = new User();
                $storedUser->login_tries = User::LOGIN_ATTEMPTS_ALLOWED;
                $storedUser->id = 0;
            }

            if (!$storedUser instanceof User)
            {
                $storedUser = new User();
                $storedUser->login_tries = User::LOGIN_ATTEMPTS_ALLOWED;
                $storedUser->id = 0;
            }

            if ($storedUser->login_tries >= User::LOGIN_ATTEMPTS_ALLOWED || $storedUser->id == null)
            {
                $this->serviceLocator->get('AuthService')->clearIdentity();
                $submissionErrors = [];
                $submissionErrors[] = 'You have exceeded the number of login attempts allowed. '.
                    'You must reset your password or have an administrator reset your password for you.';
                $viewModel->setVariable('submissionErrors', $submissionErrors);
                return $viewModel;
            }

            /****************  ******************/
            $user = new User();
            $user->exchangeArray(array(
                'username' => strtolower($username),
                'password' => $password,
                'salt' => $storedUser->salt
            ));
            $user->password = $user->encryptPassword();

            $authAdapter->getAdapter()
                ->setIdentity($user->username)
                ->setCredential($user->password);

            $result = $authAdapter->authenticate();

            if (!$result->isValid())
            {
                $storedUser->login_tries++;

                $userTable->saveOne($storedUser);

                $submissionErrors = array();
                foreach ($result->getMessages() as $message)
                {
                    $submissionErrors[] = $message;
                }

                if ($storedUser->login_tries > 1)
                {
                    $submissionErrors[] = sprintf("You have %d login attempts remaining.",
                        User::LOGIN_ATTEMPTS_ALLOWED - $storedUser->login_tries);
                }
                $viewModel->setVariable('submissionErrors', $submissionErrors);
                return $viewModel;
            }

            $user = new User();
            $user->username = $this->serviceLocator->get('AuthService')->getIdentity();

            /**
             * @var $user User
             */
            $user = $userTable->findMatchingUser($user); /* get the user by username */

            if (!$user->status)
            {
                $this->serviceLocator->get('AuthService')->clearIdentity();
                $submissionErrors = [];
                $submissionErrors[] = 'A record with the supplied identity could not be found';
                $viewModel->setVariable('submissionErrors', $submissionErrors);
                return $viewModel;
            }

            $storage = $authAdapter->getStorage();
            $storage->setRememberMe(1);

            /**
             * Once we have our user, set the two factor ok status to false so we force the
             * user to go through the two factor verification
             */
            $user->numlogins = $user->numlogins + 1;
            $user->lastlog = date("Y-m-d H:i:s");
            $user->login_tries = 0;
            $userTable->saveOne($user);

            if (isset($user->change_password) && $user->change_password)
            {
                return $this->redirect()->toUrl('/wstr/application/members/#/new-password');
            }


            return $this->redirect()->toUrl($redirect);
            /****************  ******************/
            
        }

        return $viewModel;
    }

    public function resetPasswordAction()
    {
        if ($this->serviceLocator->get('AuthService')->hasIdentity())
        {
            $this->redirect()->toUrl('/wstr/application/members/');
        }
        $this->layout('layout/logged-out');

        $config = $this->serviceLocator->get('config');
        $captcha_key = $config['google_captcha']['site_key'];
        $viewModel = new ViewModel();
        $viewModel->setVariable('captcha_key', $captcha_key);

        $key = $this->request->getQuery('key',$this->request->getPost('key', null));

        $user = null;

        $viewModel->setVariable('passwordKey', $key);

        /**
         * @var $userTable \Application\Model\UserTable
         */
        $userTable = $this->serviceLocator->get('Application\Model\UserTable');

        if ($key != null)
        {

            /**
             * @var $user \Application\Model\User
             */
            $user = $userTable->findUserByResetKey($key);
            if ($user instanceof User)
            {
                $viewModel->setVariable('hasKey', true);
            } else
            {
                $viewModel->setVariable('keyNotFound', true);
            }
        } else
        {
            $viewModel->setVariable('keyNotFound', true);
        }

        if ($this->request->getMethod() == 'POST')
        {
            $captcha_secret = $config['google_captcha']['secret_key'];
            $captcha_api    = $config['google_captcha']['api_url'];

            $captchaResponse = $this->request->getPost('g-recaptcha-response');

            $options = [
                'adapter' => Curl::class,
                'curloptions' => [
                    CURLOPT_FOLLOWLOCATION => true,
                    CURLOPT_POST => true,
                    // CURLOPT_USE_SSL => true,
                    // CURLOPT_CAINFO => '/etc/ssl/certs/ca-certificates.crt'
                ]
            ];
            new Client($captcha_api);
            $client = new Client($captcha_api, $options);
            $client->setParameterPost([
                'secret' => $captcha_secret,
                'response' => $captchaResponse,
                'remoteip' => $_SERVER['REMOTE_ADDR']
            ]);
            $client->setMethod('POST');
            $response = $client->send();

            $content = json_decode($response->getBody());

            if (!$content->success)
            {
                $submissionErrors = [];
                $submissionErrors[] = 'reCAPTCHA Anti-Robot test failed.  If you continue to experience problems please contact the help desk';
                $viewModel->setVariable('errors', $submissionErrors);
                $viewModel->setVariable('captcha_key', $captcha_key);
                return $viewModel;
            }

            $password = $this->request->getPost('password', null);
            $password_confirm = $this->request->getPost('password_confirm', null);

            if ($password != null && $password == $password_confirm && $user->passwordOkay($password))
            {
                try {
                    $user->password = $password;
                    $user->password = $user->encryptPassword();
                    $user->passwd_reset_key = null;
                    $user->passwordchangerequired = null;
                    $user->login_tries = 0;
                    $userTable->saveOne($user);
                    $viewModel->setVariable('completed', true);

                    /**
                     * We want to send a password changed notice to the admins in the system
                     * when a user changes their password
                     */
                    $changedPasswordNotificationMailer = new ChangedPasswordNotificationMailer($this->serviceLocator);
                    $changedPasswordNotificationMailer->sendPasswordChangedNotice($user);

                } catch (\Exception $e)
                {
                    $viewModel->setVariable('hasException', true);
                }
            } else
            {
                $viewModel->setVariable('hasError', true);
                $errors = array();
                if (!$user->passwordOkay($password)) $errors[] = 'Password must be at least 8 characters and contain a'.
                                ' mix of upper and lower-case letters and at least 1 special character and number';
                if ($password == null) $errors[] = 'Password is empty.';
                if ($password_confirm == null) $errors[] = 'Password verification is empty.';
                if ($password != $password_confirm) $errors[] = 'Passwords don\'t match.';
                $viewModel->setVariable('errors', $errors);
            }
        }


        return $viewModel;
    }

    public function forgotPasswordAction()
    {
        // $user = null;
        // $userTable = $this->serviceLocator->get('Application\Model\UserTable');
        // $user = $userTable->findUserByResetKey("worldwebtechnologiesprivatelimited");
        // $user->password = "foh5saeT!!";
        // $user->password = $user->encryptPassword();
        // $user->passwd_reset_key = null;
        // $user->passwordchangerequired = null;
        // $user->login_tries = 0;
        // $userTable->saveOne($user);
        // echo "<pre>"; 
        // print_r($userData);
        // echo "reset password successful."; die;

        if ($this->serviceLocator->get('AuthService')->hasIdentity())
        {
            $this->redirect()->toUrl('/wstr/application/members/');
        }
        $this->layout('layout/logged-out');

        $config = $this->serviceLocator->get('config');
        $captcha_key = $config['google_captcha']['site_key'];

        $viewModel = new ViewModel();
        $viewModel->setVariable('captcha_key', $captcha_key);


        if ($this->request->getMethod() == 'POST')
        {

            $captcha_secret = $config['google_captcha']['secret_key'];
            $captcha_api    = $config['google_captcha']['api_url'];

            $captchaResponse = $this->request->getPost('g-recaptcha-response');

            $options = [
                'adapter' => Curl::class,
                'curloptions' => [
                    CURLOPT_FOLLOWLOCATION => true,
                    CURLOPT_POST => true,
                    // CURLOPT_USE_SSL => true,
                    // CURLOPT_CAINFO => '/etc/ssl/certs/ca-certificates.crt'
                ]
            ];
            new Client($captcha_api);
            $client = new Client($captcha_api, $options);
            $client->setParameterPost([
                'secret' => $captcha_secret,
                'response' => $captchaResponse,
                'remoteip' => $_SERVER['REMOTE_ADDR']
            ]);
            $client->setMethod('POST');
            $response = $client->send();

            $content = json_decode($response->getBody());

            if (!$content->success)
            {
                $submissionErrors = [];
                $submissionErrors[] = 'reCAPTCHA Anti-Robot test failed.  If you continue to experience problems please contact the help desk';
                $viewModel->setVariable('submissionErrors', $submissionErrors);
                $viewModel->setVariable('captcha_key', $captcha_key);
                return $viewModel;
            }


            $email = $this->request->getPost('email', null);

            if ($email != null)
            {
                try
                {
                    $user = new User();
                    $user->exchangeArray(array(
                        'email' => $email
                    ));

                    /**
                     * @var $userTable \Application\Model\UserTable
                     */
                    $userTable = $this->serviceLocator->get('Application\Model\UserTable');

                    /**
                     * @var $user \Application\Model\User
                     */
                    $user = $userTable->findUserByEmail($user);

                    if (!$user instanceof User)
                    {
                        throw new EntryNotFoundException("User not found");
                    }
                    if (isset($user->id) && $user->id > 0)
                    {
                        $resetKey = md5(User::generatePassword());
                        $user->passwd_reset_key = $resetKey;
                        $userTable->saveOne($user);
                    }

                   
                    /**
                     * @var $forgotPasswordMailer ForgotPasswordMailer
                     */
                    $forgotPasswordMailer = $this->serviceLocator->get('Application\Services\Mail\ForgotPasswordMailer');
                    $forgotPasswordMailer->sendMail($user);
                    $viewModel->setVariables(array('submissionComplete' => true));

                } catch (EntryNotFoundException $notFoundException) {
                    $viewModel->setVariable('submissionComplete', true);

                } catch (\Exception $e) {
                    $viewModel->setVariable('submissionComplete', true);
                    $viewModel->setVariable('exceptionInUpdate', true);
                }
            }
        }

        return $viewModel;

    }

    public function pingAction()
    {
        $authAdapter = $this->serviceLocator->get('AuthService');

        if (!$authAdapter->hasIdentity())
        {
            return new JsonModel([ 's' => 0 ]);
        }
        $storage = $authAdapter->getStorage();
        $storage->setRememberMe(1);


        $user = $this->serviceLocator->get('ActiveUser');
        $_SESSION['_ctime'] = time();

        return new JsonModel([
            's' => 1,
            '_ctime' => time() - $_SESSION['_ctime']
        ]);
    }

}
