<?php

namespace API\Controller;

use API\Exception\AccessDeniedException;
use API\Exception\DeletedEntryNotificationException;
use API\Exception\SystemErrorException;
use Application\Exception\EntryNotFoundException;
use Application\Model\Admin;
use Application\Model\Shift;
use Application\Model\ShiftAssignment;
use Application\Model\ShiftAssignmentsTable;
use Application\Model\ShiftsTable;
use Application\Model\WeekdayAssignment;
use Application\Model\WeekdayAssignmentsTable;
use Laminas\Db\ResultSet\ResultSet;
use Laminas\Mvc\Controller\AbstractRestfulController;
use Laminas\View\Model\JsonModel;

class WeekdayAssignmentsController extends AbstractRestfulController
{
    public function get($id)
    {
        try
        {
            /**
             * @var $weekdayAssignmentsTable WeekdayAssignmentsTable
             */
            $weekdayAssignmentsTable = $this->serviceLocator->get('Application\Model\WeekdayAssignmentsTable');

            /**
             * @var $weekdayAssignment WeekdayAssignment
             */
            $weekdayAssignment = $weekdayAssignmentsTable->fetchOne($id);

            return new JsonModel($weekdayAssignment->toArray());

        } catch (EntryNotFoundException $notFoundException)
        {
            return new \API\Exception\EntryNotFoundException($this->response,
                "UNABLE TO FIND ENTRY");
        } catch (\Exception $exception)
        {
            return new JsonModel(new SystemErrorException($this->response, $exception));
        }
    }

    public function getList()
    {
        try
        {
            /**
             * @var $weekdayAssignmentsTable WeekdayAssignmentsTable
             */
            $weekdayAssignmentsTable = $this->serviceLocator->get('Application\Model\WeekdayAssignmentsTable');

            /**
             * @var $resultSet ResultSet
             */
            $resultSet = $weekdayAssignmentsTable->fetchAll();

            $return = [];

            /**
             * @var $weekdayAssignment WeekdayAssignment
             */
            while ($weekdayAssignment = $resultSet->current())
            {
                $return[] = $weekdayAssignment->toArray();
                $resultSet->next();
            }

            return new JsonModel($return);

        } catch (EntryNotFoundException $notFoundException)
        {
            return new \API\Exception\EntryNotFoundException($this->response,
                "UNABLE TO FIND ENTRY");
        } catch (\Exception $exception)
        {
            return new JsonModel(new SystemErrorException($this->response, $exception));
        }
    }

    public function create($data)
    {
        try
        {
            /**
             * @var $activeAdmin Admin
             */
            $activeAdmin = $this->serviceLocator->get('ActiveAdmin');

            if (!$activeAdmin->enabled)
            {
                return new JsonModel(new AccessDeniedException($this->response, 'Administrative Access Required'));
            }

            /**
             * @var $weekdayAssignmentsTable WeekdayAssignmentsTable
             */
            $weekdayAssignmentsTable = $this->serviceLocator->get('Application\Model\WeekdayAssignmentsTable');

            $weekdayAssignment = new WeekdayAssignment();
            $weekdayAssignment->exchangeArray($data);
            $weekdayAssignment->id = null;

            $weekdayAssignmentsTable->saveOne($weekdayAssignment);

            /**************************************************************************\
             * After creating a new weekday assignment, we need to find the shifts
             * assigned to the daytype so we can create a new shiftassignment for each
             * shift in the weekdayassignment
             *
            \**************************************************************************/

            /**
             * @var $shiftsTable ShiftsTable
             */
            $shiftsTable = $this->serviceLocator->get('Application/Model/ShiftsTable');

            /**
             * @var $shiftsResultSet ResultSet
             */
            $shiftsResultSet = $shiftsTable->fetchShiftsByDayType($weekdayAssignment->daytype_id);

            /**
             * @var $shiftAssignmentsTable ShiftAssignmentsTable
             */
            $shiftAssignmentsTable = $this->serviceLocator->get('Application\Model\ShiftAssignmentsTable');


            /**
             * @var $shift Shift
             */
            while ($shift = $shiftsResultSet->current())
            {
                $shiftAssignment = new ShiftAssignment();
                $shiftAssignment->weekdayassignment_id = $weekdayAssignment->id;
                $shiftAssignment->shift_id = $shift->id;

                $shiftAssignmentsTable->saveOne($shiftAssignment);

                $shiftsResultSet->next();
            }

            return new JsonModel($weekdayAssignment->toArray());

        } catch (\Exception $exception)
        {
            return new JsonModel(new SystemErrorException($this->response, $exception));
        }
    }

    public function update($id, $data)
    {
        try
        {

            /**
             * @var $activeAdmin Admin
             */
            $activeAdmin = $this->serviceLocator->get('ActiveAdmin');

            if (!$activeAdmin->enabled)
            {
                return new JsonModel(new AccessDeniedException($this->response, 'Administrative Access Required'));
            }

            /**
             * @var $weekdayAssignmentsTable WeekdayAssignmentsTable
             */
            $weekdayAssignmentsTable = $this->serviceLocator->get('Application\Model\WeekdayAssignmentsTable');

            /**
             * @var $oldAssignment WeekdayAssignment
             */
            $oldAssignment = $weekdayAssignmentsTable->fetchOne($id);

            $weekdayAssignment = new WeekdayAssignment();
            $weekdayAssignment->exchangeArray($data);
            $weekdayAssignment->id = $id;

            $recreateShiftAssignments = ($oldAssignment->daytype_id != $weekdayAssignment->daytype_id);

            $weekdayAssignmentsTable->saveOne($weekdayAssignment);

            /**************************************************************************\
             * When changing the day type for the season's weekday assignment, we need
             * to update the shift assignments mapping
            \**************************************************************************/

            if ($recreateShiftAssignments)
            {
                /**
                 * @var $shiftAssignmentsTable ShiftAssignmentsTable
                 */
                $shiftAssignmentsTable = $this->serviceLocator->get('Application\Model\ShiftAssignmentsTable');

                $shiftAssignmentsTable->deleteMatching($weekdayAssignment);

                /**
                 * @var $shiftsTable ShiftsTable
                 */
                $shiftsTable = $this->serviceLocator->get('Application/Model/ShiftsTable');

                /**
                 * @var $shiftsResultSet ResultSet
                 */
                $shiftsResultSet = $shiftsTable->fetchShiftsByDayType($weekdayAssignment->daytype_id);


                /**
                 * @var $shift Shift
                 */
                while ($shift = $shiftsResultSet->current())
                {
                    $shiftAssignment = new ShiftAssignment();
                    $shiftAssignment->weekdayassignment_id = $weekdayAssignment->id;
                    $shiftAssignment->shift_id = $shift->id;

                    $shiftAssignmentsTable->saveOne($shiftAssignment);

                    $shiftsResultSet->next();
                }
            }

            return new JsonModel($weekdayAssignment->toArray());

        } catch (EntryNotFoundException $notFoundException)
        {
            return new \API\Exception\EntryNotFoundException($this->response,
                "UNABLE TO FIND ENTRY");
        } catch (\Exception $exception)
        {
            return new JsonModel(new SystemErrorException($this->response, $exception));
        }
    }

    public function delete($id)
    {
        try {

            /**
             * @var $activeAdmin Admin
             */
            $activeAdmin = $this->serviceLocator->get('ActiveAdmin');

            if (!$activeAdmin->enabled)
            {
                return new JsonModel(new AccessDeniedException($this->response, 'Administrative Access Required'));
            }

            /**
             * @var $weekdayAssignmentsTable WeekdayAssignmentsTable
             */
            $weekdayAssignmentsTable = $this->serviceLocator->get('Application\Model\WeekdayAssignmentsTable');

            $weekdayAssignmentsTable->deleteOne($id);

            return new JsonModel(new DeletedEntryNotificationException($this->response, "Assignment Deleted"));

        } catch (EntryNotFoundException $notFoundException)
        {
            return new \API\Exception\EntryNotFoundException($this->response,
                "UNABLE TO FIND ENTRY");
        } catch (\Exception $exception)
        {
            return new JsonModel(new SystemErrorException($this->response, $exception));
        }

    }
}
