<?php

namespace API\Controller;

use API\Exception\AccessDeniedException;
use API\Exception\DeletedEntryNotificationException;
use API\Exception\SystemErrorException;
use Application\Exception\EntryNotFoundException;
use Application\Model;
use Application\Services\Mail\ChangedPasswordNotificationMailer;
use Laminas\Db\ResultSet\ResultSet;
use Laminas\Mvc\Controller\AbstractRestfulController;
use Laminas\View\Model\JsonModel;

class UserController extends AbstractRestfulController
{
    public function get($id)
    {
        if ($id === 'all')
        {
            return $this->getList(true);
        }

        try
        {

            /**
             * @var $activeUser Model\User
             */
            $activeUser = $this->serviceLocator->get('ActiveUser');
            $activeUser->salt = null;
            $activeUser->password = null;
            $activeUser->passwd_reset_key = null;

            /**
             * @var $activeAdmin Model\Admin
             */
            $activeAdmin = $this->serviceLocator->get('ActiveAdmin');

            if ($id == 'active')
            {
                $userArray = $activeUser->toArray();
                if ($activeAdmin->enabled)
                {
                    $userArray['is_admin'] = 1;
                }
                return new JsonModel($userArray);
            }


            /**
             * @var $userTable Model\UserTable
             */
            $userTable = $this->serviceLocator->get('Application\Model\UserTable');

            /**
             * @var $adminTable Model\AdminTable
             */
            $adminTable = $this->serviceLocator->get('Application\Model\AdminTable');

            if ( preg_match('/[a-zA-Z]/', $id) )
            {
                $user = new Model\User();
                $user->email = $id;

                /**
                 * @var $user Model\User
                 */
                $user = $userTable->findMatchingUser($user);
            } else
                {

                /**
                 * @var $user Model\User
                 */
                $user = $userTable->fetchOne((int) $id);
            }


            $loginAs = $this->request->getQuery('loginAs', false);

            if ($loginAs && $activeAdmin->enabled)
            {

                $authAdapter = $this->serviceLocator->get('AuthService');
                $authAdapter->getAdapter()
                    ->setIdentity($user->username)
                    ->setCredential($user->password);
                $result = $authAdapter->authenticate();

                if (!$result->isValid())
                {
                    return new JsonModel(new AccessDeniedException($this->request, 'Unable to authenticate'));
                }

                return new JsonModel(['authentication' => 'success']);
            }


            $user->salt = null;
            $user->password = null;

//            $adminModel = $adminTable->fetchOne($user->id);

            $userArray = $user->toArray();


            return new JsonModel($userArray);

        } catch (EntryNotFoundException $notFoundException)
        {
            return new JsonModel(new \API\Exception\EntryNotFoundException($this->response,
                "Unable to find the requested User"));
        } catch (\Exception $exception)
        {
            return new JsonModel(new SystemErrorException($this->response, $exception));
        }
    }

    public function getList()
    {

        try
        {
            $returnData = [];

            /**
             * @var $userTable Model\UserTable
             */
            $userTable = $this->serviceLocator->get('Application\Model\UserTable');

            /**
             * @var $adminTable Model\AdminTable
             */
            $adminTable = $this->serviceLocator->get('Application\Model\AdminTable');

            /**
             * @var $users ResultSet
             */
            $users = $userTable->fetchAll();

            /**
             * @var $user Model\User
             */
            while (($user = $users->current()))
            {
                $user->salt = null;
                $user->password = null;
                $user->passwd_reset_key = null;

                $userArray = $user->toArray();

                /**
                 * @var $admin Model\Admin
                 */
                $admin = $adminTable->fetchOne($user->id);
                $userArray['is_admin'] = $admin->enabled;

                $returnData[] = $userArray;

                $users->next();
            }

            return new JsonModel($returnData);
        } catch (EntryNotFoundException $notFoundException)
        {
            return new JsonModel(new \API\Exception\EntryNotFoundException($this->response,
                "Unable to find any users for the current business"));
        } catch (\Exception $exception)
        {
            return new JsonModel(new SystemErrorException($this->response, $exception));
        }

    }

    public function create($data)
    {
        try
        {
            $newUser = new Model\User();
            $newUser->exchangeArray($data);


            /**
             * @var $activeAdmin Model\Admin
             */
            $activeAdmin = $this->serviceLocator->get('ActiveAdmin');

            /**
             * @var $userTable Model\UserTable
             */
            $userTable = $this->serviceLocator->get('Application\Model\UserTable');

            if (!$activeAdmin->enabled)
            {
                return new JsonModel(new AccessDeniedException($this->response,
                    "Unable to add a new user: You must be an administrator"));
            }

            if (in_array(null, [$newUser->name, $newUser->username, $newUser->email]))
            {
                return new JsonModel(new AccessDeniedException($this->response,
                    "Unable to create new user: All of Name, Username and Email address must be supplied"));
            }

            $salt = Model\User::createSalt();

            if (isset($data['new_password']) && isset($data['password_verify']) &&
                $newUser->passwordOkay($data['new_password']) && $data['new_password'] == $data['password_verify']
            )
            {
                $gen_passwd = $data['new_password'];
                $newUser->salt = $salt;
                $newUser->password = $newUser->encryptPassword($gen_passwd);
            } else
            {
                xdebug_break();
                if (isset($data['new_password']) && isset($data['password_verify']) &&
                    $data['new_password'] == $data['password_verify'])
                {
                    return new JsonModel(new AccessDeniedException($this->response, 'Password must be at least 8 characters and contain a' .
                        ' mix of upper and lower-case letters and at least 1 special character and number'));
                } else
                {
                    return new JsonModel(new AccessDeniedException($this->response, "Passwords don't match"));
                }
            }

            $foundUser = $userTable->findMatchingUser($newUser);
            $foundUser2 = $userTable->findUserByEmail($newUser);

            if ($foundUser instanceof Model\User || $foundUser2 instanceof  Model\User)
            {
                $dup_field = '';
                if ($foundUser instanceof Model\User)
                {
                    $dup_field = 'user name';
                }
                if ($foundUser2 instanceof Model\User)
                {
                    $dup_field .= (strlen($dup_field) ? ' and ' : '') . 'email address';
                }
                return new JsonModel(new AccessDeniedException($this->response,
                    sprintf("Unable to create new user, the %s submitted is already in use.", $dup_field)));
            }

            /**
             * ok, so we've passed all checks - let's go ahead and save changes to the user
             */

            $newUser->id = null;
            $newUser->numlogins = 0;
            $newUser->lastlog = null;
            $newUser->create_date = date("Y-m-d");
            $newUser->modify_date = date("Y-m-d");
            $newUser->status = (int) $newUser->status;
            $newUser->organization_constitutional = (int) $newUser->organization_constitutional;
            $newUser->organization_spirits = (int) $newUser->organization_spirits;
            $userTable->saveOne($newUser);

        } catch (\Exception $exception)
        {
            return new JsonModel(new SystemErrorException($this->response, $exception));
        }

        /**
         * remember to clear out the password so we don't accidentally return an encrypted
         * or (worse) unencrypted password to the end-user
         */
        $newUser->salt = null;
        $newUser->password = null;

        return new JsonModel($newUser->toArray());

    }

    public function update($id, $data)
    {

        try
        {

            /**
             * @var $userTable Model\UserTable
             */
            $userTable = $this->serviceLocator->get('Application\Model\UserTable');

            /**
             * @var $activeUser Model\User
             */
            $activeUser = $this->serviceLocator->get('ActiveUser');


            /**
             * @var $activeAdmin Model\Admin
             */
            $activeAdmin = $this->serviceLocator->get('ActiveAdmin');


            /**
             * @var $oldUser Model\User
             */
            $oldUser = $userTable->fetchOne((int)$id);

            $newUser = new Model\User();
            $newUser->exchangeArray($data);
            $newUser->id = (int)$id;

            $userMatchingEmail = $userTable->findUserByEmail($newUser);
            $userMatchingUsername = $userTable->findMatchingUser($newUser);

            if ((int) $activeUser->id != (int) $oldUser->id && !$activeAdmin->enabled)
            {
                return new JsonModel(new AccessDeniedException($this->response,
                    "You do not have permission to modify user accounts"));
            }


            $new_password = array_key_exists('new_password', $data) ? $data['new_password'] : '';
            $password_verify = array_key_exists('password_verify', $data) ? $data['password_verify'] : '';

            $np_len = strlen($new_password);
            $pv_len = strlen($password_verify);


            if ($userMatchingEmail instanceof Model\User && (int) $userMatchingEmail->id !== (int) $newUser->id)
            {
                return new JsonModel(new AccessDeniedException($this->response,
                    "Unable to change email address to the one submitted.  There is already a user".
                    "with that email address in the system."));
            }
            if ($userMatchingUsername instanceof Model\User && (int) $userMatchingUsername->id !== (int) $newUser->id)
            {
                return new JsonModel(New AccessDeniedException($this->response,
                    "Unable to change username to the one submitted. There is already a user ".
                    "with that username in the system."));
            }

            /**
             * We're just updating ourselves.
             */
            if ((int)$newUser->id === (int)$activeUser->id)
            {
                if (($np_len + $pv_len) > 0)
                {
                    $newUser->salt = $activeUser->salt;
                    $newUser->password = $newUser->encryptPassword($newUser->password);

                    if ($newUser->password != $activeUser->password)
                    {
                        return new JsonModel(new AccessDeniedException($this->response,
                            "The password submitted does not match your current password."));
                    }

                    if ($new_password != $password_verify)
                    {
                        return new JsonModel(new AccessDeniedException($this->response,
                            "Your new password and confirmation password do not match."));
                    }

                    if (!$newUser->passwordOkay($new_password))
                    {
                        return new JsonModel(new AccessDeniedException($this->response,
                            'Password must be at least 8 characters and contain a mix of upper '.
                            'and lower-case letters and at least 1 special character and number'
                        ));
                    }

                    /**
                     * When setting a new password, generate a new unique salt
                     */
                    $newUser->salt = Model\User::createSalt();
                    $newUser->password = $data['new_password'];
                    $newUser->password = $newUser->encryptPassword();
                    $newUser->passwordchangerequired = 0;

                    /**
                     * We want to send a password changed notice to the admins in the system
                     * when a user changes their password
                     */
                    if ($newUser->id !== $activeAdmin->enabled)
                    {
                        $changedPasswordNotificationMailer = new ChangedPasswordNotificationMailer($this->serviceLocator);
                        $changedPasswordNotificationMailer->sendPasswordChangedNotice($newUser);
                    }

                } else
                {
                    $newUser->salt = $activeUser->salt;
                    $newUser->password = $activeUser->password;
                }


                $newUser->create_date = $oldUser->create_date;
                $newUser->modify_date = date("Y-m-d H:i:s");
                $newUser->lastlog = $oldUser->lastlog;
                $userTable->saveOne($newUser);

                if ($newUser->username != $oldUser->username)
                {
                    $authAdapter = $this->serviceLocator->get('AuthService');
                    $authAdapter->getAdapter()
                        ->setIdentity($newUser->username)
                        ->setCredential($newUser->password);
                }

                $newUser->salt = null;
                $newUser->password = null;

                return new JsonModel($newUser->toArray());
            }


            if (($np_len + $pv_len) > 0)
            {
                if ($new_password != $password_verify)
                {
                    return new JsonModel(new AccessDeniedException($this->response,
                        "The new password and confirmation password do not match."));
                }

                if (!$newUser->passwordOkay($new_password))
                {
                    return new JsonModel(new AccessDeniedException($this->response,
                        'Password must be at least 8 characters and contain a mix of upper '.
                        'and lower-case letters and at least 1 special character and number'
                    ));
                }

                $newUser->salt = Model\User::createSalt();
                $newUser->password = $data['new_password'];
                $newUser->password = $newUser->encryptPassword();
            } else
            {
                $newUser->password = $oldUser->password;
            }

            $newUser->lastlog = $oldUser->lastlog;
            $newUser->create_date = $oldUser->create_date;
            $newUser->modify_date = date("Y-m-d H:i:s");
            $newUser->passwordchangerequired = $oldUser->passwordchangerequired;
            $newUser->passwd_reset_key = null;

            $newUser->lastlog = $userMatchingUsername->lastlog;
            $userTable->saveOne($newUser);

            /**
             * remember to clear out the password so we don't accidentally return an encrypted
             * or (worse) unencrypted password to the end-user
             */
            $newUser->salt = $oldUser->salt = null;
            $newUser->password = null;

            return new JsonModel($newUser->toArray());

        } catch (EntryNotFoundException $notFoundException)
        {
            return new JsonModel(new \API\Exception\EntryNotFoundException($this->response,
                "Unable to update the requested user: user not found"));
        } catch (\Exception $exception)
        {
            return new JsonModel(new SystemErrorException($this->response, $exception));
        }
    }


    public function delete($id)
    {
        try
        {

            /**
             * @var $activeAdmin Model\Admin
             */
            $activeAdmin = $this->serviceLocator->get('ActiveAdmin');

            if (!$activeAdmin->enabled)
            {
                return new JsonModel(new AccessDeniedException($this->response, "Unable to delete user: you are not an admin"));
            }

            /**
             * @var $userTable Model\UserTable
             */
            $userTable = $this->serviceLocator->get('Application\Model\UserTable');

            $userTable->deleteOne($id);

            return new JsonModel(new DeletedEntryNotificationException($this->response, "User deleted"));

        } catch (EntryNotFoundException $notFoundException)
        {
            return new JsonModel(new \API\Exception\EntryNotFoundException($this->response,
                "Unable to find the requested User"));
        } catch (\Exception $exception)
        {
            return new JsonModel(new SystemErrorException($this->response, $exception));
        }
    }
}
