<?php

namespace API\Controller;

use API\Exception\AccessDeniedException;
use API\Exception\DeletedEntryNotificationException;
use API\Exception\SystemErrorException;
use Application\Exception\EntryNotFoundException;
use Application\Model\Admin;
use Application\Model\GuideAssignment;
use Application\Model\GuideAssignmentTable;
use Application\Model\GuideShiftAssignment;
use Application\Model\GuideShiftAssignmentTable;
use Application\Model\Season;
use Application\Model\SeasonsTable;
use Application\Model\SeasonTaskAssignmentMap;
use Application\Model\SeasonTaskAssignmentMapTable;
use Application\Model\ShiftAssignment;
use Application\Model\ShiftAssignmentsTable;
use Application\Model\Task;
use Application\Model\TaskAssignment;
use Application\Model\TaskAssignmentTable;
use Application\Model\TasksTable;
use Application\Model\WeekdayAssignment;
use Application\Model\WeekdayAssignmentsTable;
use Application\Services\DatePickerFormatter;
use Laminas\Db\ResultSet\ResultSet;
use Laminas\Mvc\Controller\AbstractRestfulController;
use Laminas\View\Model\JsonModel;

class SeasonsController extends AbstractRestfulController
{
    public function get($id)
    {
        try
        {

            /**
             * @var $seasonsTable SeasonsTable
             */
            $seasonsTable = $this->serviceLocator->get('Application/Model/SeasonsTable');

            /**
             * @var $season Season
             */
            $season = $seasonsTable->fetchOne($id);
            $season->startdate = DatePickerFormatter::toDatePickerDate($season->startdate);
            $season->enddate   = DatePickerFormatter::toDatePickerDate($season->enddate);

            return new JsonModel($season->toArray());

        } catch (EntryNotFoundException $notFoundException)
        {
            return new \API\Exception\EntryNotFoundException($this->response,
                "UNABLE TO FIND ENTRY");
        } catch (\Exception $exception)
        {
            return new JsonModel(new SystemErrorException($this->response, $exception));
        }
    }

    public function getList()
    {
        try
        {
            /**
             * @var $seasonsTable SeasonsTable
             */
            $seasonsTable = $this->serviceLocator->get('Application/Model/SeasonsTable');

            /**
             * @var $resultSet ResultSet
             */
            $resultSet = $seasonsTable->fetchOrdered();

            $return = [];

            /**
             * @var $season Season
             */
            while ($season = $resultSet->current())
            {
                $season->startdate = DatePickerFormatter::toDatePickerDate($season->startdate);
                $season->enddate   = DatePickerFormatter::toDatePickerDate($season->enddate);
                $return[] = $season->toArray();
                $resultSet->next();
            }

            return new JsonModel($return);

        } catch (EntryNotFoundException $notFoundException)
        {
            return new \API\Exception\EntryNotFoundException($this->response,
                "UNABLE TO FIND ENTRY");
        } catch (\Exception $exception)
        {
            return new JsonModel(new SystemErrorException($this->response, $exception));
        }
    }

    public function create($data)
    {
        try
        {

            /**
             * @var $activeAdmin Admin
             */
            $activeAdmin = $this->serviceLocator->get('ActiveAdmin');

            if (!$activeAdmin->enabled)
            {
                return new JsonModel(new AccessDeniedException($this->response, 'Administrative Access Required'));
            }


            /**
             * @var $seasonsTable SeasonsTable
             */
            $seasonsTable = $this->serviceLocator->get('Application/Model/SeasonsTable');

            $season = new Season();
            $season->exchangeArray($data);
            $season->id = null;
            $season->status = 1;
            $season->organization_id = 1;
            $season->startdate = date('Y-m-d');
            $season->enddate = date('Y-m-d');
            $season->createdate = date('Y-m-d H:i:s');
            $season->modifydate = date('Y-m-d H:i:s');

            $seasonsTable->saveOne($season);

            return new JsonModel($season->toArray());

        } catch (\Exception $exception)
        {
            return new JsonModel(new SystemErrorException($this->response, $exception));
        }
    }

    private function setSeasonTasks(Season $season, $deleteOld)
    {

        /**
         * @var $seasonTaskAssignmentMapTable SeasonTaskAssignmentMapTable
         */
        $seasonTaskAssignmentMapTable = $this->serviceLocator->get('Application\Model\SeasonTaskAssignmentMapTable');

        /**
         * @var $taskAssignmentTable TaskAssignmentTable
         */
        $taskAssignmentTable = $this->serviceLocator->get('Application\Model\TaskAssignmentTable');

        /**
         * @var $guideShiftAssignmentTable GuideShiftAssignmentTable
         */
        $guideShiftAssignmentTable = $this->serviceLocator->get('Application\Model\GuideShiftAssignmentTable');

        /**
         * @var $guideAssignmentTable GuideAssignmentTable
         */
        $guideAssignmentTable = $this->serviceLocator->get('Application\Model\GuideAssignmentTable');


        /**********************************************************************************************************\
         * all this just to find the list of tasks that are assigned to the shifts that are assigned to the
         * weekday assignments that are assigned to the season, so that we can the check to see if the task has
         * been created in taskassignments (and possibly guideassignments if a guide has been assigned to a shift)
         * and if not, create the taskassignment (and possibly guideassignment)
        \**********************************************************************************************************/


        /**
         * @var $seasonTaskAssignmentMapResultSet ResultSet
         */
        $seasonTaskAssignmentMapResultSet = $seasonTaskAssignmentMapTable->getSeasonTaskMap($season->id);


        $seasonTaskAssignmentsMaps = [];
        $taskAssignments = [];
        $taskAssignmentIds = [];
        $guideShiftAssignments = [];
        $guideAssignments = [];

        /**************************************************************************************************************
         * Iterate once over the set of season tasks so we can save them off and so we can build a map of
         * existing task assignments, guide shift assignments and later guide assignments
         *
         * This will save a TON of time later rather than doing direct SQL query lookups for each task assignment,
         * simply by pulling the entire season task assignment map, tasks assignments and creating a O(1) lookup table
         * I've reduced the time from 20+s to ~5s when only creating new assignments
         *************************************************************************************************************/

         /**
         * @var $seasonTaskAssignmentMap SeasonTaskAssignmentMap
         */
        while ($seasonTaskAssignmentMap = $seasonTaskAssignmentMapResultSet->current())
        {
            $seasonTaskAssignmentsMaps[] = $seasonTaskAssignmentMap;

            if ($deleteOld)
            {
                $taskAssignmentTable->deleteMatching($season->organization_id == 1 ? 2 : 1, $seasonTaskAssignmentMap->shiftassignment_id);
            }

            /**
             * @var $taskAssignmentsResultSet ResultSet
             */
            $taskAssignmentsResultSet = $taskAssignmentTable->fetchMatching($season->organization_id, $seasonTaskAssignmentMap->shiftassignment_id);

            /**
             * @var $taskAssignment TaskAssignment
             */
            while ($taskAssignment = $taskAssignmentsResultSet->current())
            {
                /**
                 * We hash the these values so that later we can use an O(1) lookup
                 */
                $key = md5(sprintf("%d_%d_%d_%s",  $taskAssignment->organization_id,
                    $taskAssignment->shiftassignment_id, $taskAssignment->task_id, $taskAssignment->assignmentdate));

                $taskAssignments[$key] = $taskAssignment;

                $taskAssignmentIds[] = $taskAssignment->id;
                $taskAssignmentsResultSet->next();
            }

            /**
             * @var $guideShiftAssignmentsResultSet ResultSet
             */
            $guideShiftAssignmentsResultSet = $guideShiftAssignmentTable->fetchAllByShiftAssignmentId($seasonTaskAssignmentMap->shiftassignment_id);

            /**
             * @var $guideShiftAssignment GuideShiftAssignment
             */
            while ($guideShiftAssignment = $guideShiftAssignmentsResultSet->current())
            {
                $key = md5(sprintf("%d_%d_%s",
                    $guideShiftAssignment->guide_id,
                    $guideShiftAssignment->shiftassignment_id,
                    $guideShiftAssignment->assignmentdate
                ));
                $guideShiftAssignments[$key] = $guideShiftAssignment;
                $guideShiftAssignmentsResultSet->next();
            }

            $seasonTaskAssignmentMapResultSet->next();
        }

        /**********************************************************************************************************
         * Next step, find any guide assignments that are attached to the list of task assignments we found
         * earlier and save them off to a O(1) map
         *********************************************************************************************************/

        /**
         * @var $guideAssignmentsResultSet ResultSet
         */
        $guideAssignmentsResultSet = $guideAssignmentTable->fetchAllByAssignmentIds($taskAssignmentIds, true);

        /**
         * @var $guideAssignment GuideAssignment
         */
        while ($guideAssignment = $guideAssignmentsResultSet->current())
        {
            $ga_key = md5(sprintf("%d_%d", $guideAssignment->guide_id, $guideAssignment->taskassignment_id));
            $guideAssignments[$ga_key] = $guideAssignment;

            $guideAssignmentsResultSet->next();
        }


        /**********************************************************************************************************
         * Final phase:
         *
         * For each day starting at the season start date and ending on the season end date,
         *
         * 1) search to see if a task assignment has already been created; if not create one
         * 2) (if guide assigned) search to see if a guide shift assignment has been created; if not create one
         * 3) (if guide assigned) search to see if a guide assignment attached to task assignment; if not create one
         */

        /**
         * For every day in the season
         */
        $date = strtotime($season->startdate);
        $end = strtotime($season->enddate);

        while ($date <= $end)
        {
            $dayofweek = date('w', $date);

            foreach ($seasonTaskAssignmentsMaps as $seasonTaskAssignmentMap)
            {
                /**
                 * If this task should not be assigned for this day of the week, go to the next
                 */
                if ($dayofweek != $seasonTaskAssignmentMap->dayofweek) continue;

                /**
                 * If task assignment isn't found, create one
                 *
                 */
                $task_key = md5(sprintf("%d_%d_%d_%s",  $seasonTaskAssignmentMap->organization_id,
                    $seasonTaskAssignmentMap->shiftassignment_id, $seasonTaskAssignmentMap->task_id,
                    date("Y-m-d", $date)));

                if (!array_key_exists($task_key, $taskAssignments))
                {
                    $taskAssignment = new TaskAssignment();
                    $taskAssignment->shiftassignment_id = $seasonTaskAssignmentMap->shiftassignment_id;
                    $taskAssignment->guide_id = $seasonTaskAssignmentMap->guide_id ?: null;
                    $taskAssignment->organization_id = $season->organization_id;
                    $taskAssignment->task_id = $seasonTaskAssignmentMap->task_id;
                    $taskAssignment->tasktype = $seasonTaskAssignmentMap->tasktype;
                    $taskAssignment->tourtype = $taskAssignment->tasktype == 1 ? 1 : null;
                    $taskAssignment->assignmentdate = date("Y-m-d", $date);
                    $taskAssignment->starttime = $seasonTaskAssignmentMap->starttime;
                    $taskAssignment->endtime = $seasonTaskAssignmentMap->endtime;
                    $taskAssignment->status = 1;
                    $taskAssignment->createdate = date("Y-m-d H:i:s");
                    $taskAssignment->modifydate = date("Y-m-d H:i:s");

                    $taskAssignmentTable->saveOne($taskAssignment);
                } else
                {
                    $taskAssignment = $taskAssignments[$task_key];
                }

                /**
                 * if the shift assignment is assigned to a guide and there is no guide assigned to the task assignment,
                 * create a new guide assignment
                 */
                if ($seasonTaskAssignmentMap->guide_id)
                {

                    $gsa_key = md5(sprintf("%d_%d_%s",
                        $seasonTaskAssignmentMap->guide_id,
                        $seasonTaskAssignmentMap->shiftassignment_id,
                        date("Y-m-d", $date)
                    ));

                    if (!array_key_exists($gsa_key, $guideShiftAssignments))
                    {
                        $guideShiftAssignment = new GuideShiftAssignment();
                        $guideShiftAssignment->shiftassignment_id = $seasonTaskAssignmentMap->shiftassignment_id;;
                        $guideShiftAssignment->guide_id = $seasonTaskAssignmentMap->guide_id;;
                        $guideShiftAssignment->assignmentdate = $taskAssignment->assignmentdate;

                        try {
                            $guideShiftAssignmentTable->saveOne($guideShiftAssignment);
                        } catch (\Exception $e)
                        {
                            $stop = true;
                        }
                    }

                    $ga_key = md5(sprintf("%d_%d", $seasonTaskAssignmentMap->guide_id, $taskAssignment->id));

                    if (!array_key_exists($ga_key, $guideAssignments))
                    {
                        $guideAssignment = new GuideAssignment();
                        $guideAssignment->guide_id = $seasonTaskAssignmentMap->guide_id;
                        $guideAssignment->taskassignment_id = $taskAssignment->id;
                        $guideAssignment->createdate = date("Y-m-d H:i:s");
                        $guideAssignment->modifydate = date("Y-m-d H:i:s");
                        $guideAssignment->status = 1;

                        $guideAssignmentTable->saveOne($guideAssignment);
                    }

                }

            }

            $date += 86400; /* add 1 day's worth of seconds... */

        }

    }

    private function cloneSeason($id, $data)
    {
        /**
         * @var $oldSeason Season
         */
        $oldSeason = null;
        $newSeason = new Season();
        $newSeason->exchangeArray($data);

        /**
         * @var $seasonsTable SeasonsTable
         */
        $seasonsTable = $this->serviceLocator->get('Application/Model/SeasonsTable');

        /**
         * @var $weekdayAssignmentsTable WeekdayAssignmentsTable
         */
        $weekdayAssignmentsTable = $this->serviceLocator->get('Application\Model\WeekdayAssignmentsTable');

        /**
         * @var $shiftAssignmentsTable ShiftAssignmentsTable
         */
        $shiftAssignmentsTable = $this->serviceLocator->get('Application\Model\ShiftAssignmentsTable');


        try {

            $oldSeason = $seasonsTable->fetchOne($newSeason->id);

        } catch (EntryNotFoundException $notFoundException)
        {
            return new JsonModel(new \API\Exception\EntryNotFoundException($this->response,
                "Unable to clone season"));
        } catch (\Exception $exception)
        {
            return new JsonModel(new SystemErrorException($this->response, $exception));
        }

        $newSeason->id = null;

        try {
            $newSeason->startdate = DatePickerFormatter::toSqlDate($newSeason->startdate);
            $newSeason->enddate   = DatePickerFormatter::toSqlDate($newSeason->enddate);
            $newSeason->createdate = date("Y-m-d");
            $newSeason->modifydate = date("Y-m-d");

            $seasonsTable->saveOne($newSeason);

            $newSeason->startdate = DatePickerFormatter::toDatePickerDate($newSeason->startdate);
            $newSeason->enddate = DatePickerFormatter::toDatePickerDate($newSeason->enddate);

        } catch (\Exception $exception)
        {
            return new JsonModel(new SystemErrorException($this->response, $exception));
        }

        try
        {
            /**
             * @var $weekdayAssignmentResultSet ResultSet
             */
            $weekdayAssignmentResultSet = $weekdayAssignmentsTable->fetchBySeason($oldSeason);

            /**
             * @var $weekdayAssignment WeekdayAssignment
             */
            while ($weekdayAssignment = $weekdayAssignmentResultSet->current())
            {
                $newWeekDayAssignment = new WeekdayAssignment();
                $newWeekDayAssignment->season_id = $newSeason->id;
                $newWeekDayAssignment->dayofweek = $weekdayAssignment->dayofweek;
                $newWeekDayAssignment->daytype_id = $weekdayAssignment->daytype_id;

                $weekdayAssignmentsTable->saveOne($newWeekDayAssignment);

                /**
                 * @var $shiftAssignmentsResultSet ResultSet
                 */
                $shiftAssignmentsResultSet = $shiftAssignmentsTable->fetchList($weekdayAssignment->id, null);

                /**
                 * @var $shiftAssignment ShiftAssignment
                 */
                while ($shiftAssignment = $shiftAssignmentsResultSet->current())
                {
                    $newShiftAssignment = new ShiftAssignment();
                    $newShiftAssignment->weekdayassignment_id = $newWeekDayAssignment->id;
                    $newShiftAssignment->shift_id = $shiftAssignment->shift_id;

                    $shiftAssignmentsTable->saveOne($newShiftAssignment);

                    $shiftAssignmentsResultSet->next();
                }

                $weekdayAssignmentResultSet->next();
            }
        } catch (\Exception $exception)
        {
            return new JsonModel(new SystemErrorException($this->response, $exception));
        }

        $this->setSeasonTasks($newSeason);
        return new JsonModel($newSeason->toArray());
    }

    public function update($id, $data)
    {
        try
        {

            /**
             * @var $activeAdmin Admin
             */
            $activeAdmin = $this->serviceLocator->get('ActiveAdmin');


            if (!$activeAdmin->enabled)
            {
                return new JsonModel(new AccessDeniedException($this->response, 'Administrative Access Required'));
            }

            if ($id === 'clone')
            {
                return $this->cloneSeason($id, $data);
            }

            /**
             * @var $seasonsTable SeasonsTable
             */
            $seasonsTable = $this->serviceLocator->get('Application/Model/SeasonsTable');

            $toUpdateSeason = new Season();
            $toUpdateSeason->exchangeArray($data);

            /**
             * @var $oldSeason Season
             */
            $oldSeason = $seasonsTable->fetchOne($id);

            $toUpdateSeason->id = $oldSeason->id;
            $toUpdateSeason->createdate = $oldSeason->createdate;
            $toUpdateSeason->modifydate = date('Y-m-d H:i:s');

            $toUpdateSeason->startdate = DatePickerFormatter::toSqlDate($toUpdateSeason->startdate);
            $toUpdateSeason->enddate   = DatePickerFormatter::toSqlDate($toUpdateSeason->enddate);

            $seasonsTable->saveOne($toUpdateSeason);

            $this->setSeasonTasks($toUpdateSeason, $oldSeason->organization_id !== $toUpdateSeason->organization_id);

            $toUpdateSeason->startdate = DatePickerFormatter::toDatePickerDate($toUpdateSeason->startdate);
            $toUpdateSeason->enddate = DatePickerFormatter::toDatePickerDate($toUpdateSeason->enddate);

            return new JsonModel($toUpdateSeason->toArray());

        } catch (EntryNotFoundException $notFoundException)
        {
            return new JsonModel(new \API\Exception\EntryNotFoundException($this->response,
                "UNABLE TO FIND ENTRY"));
        } catch (\Exception $exception)
        {
            return new JsonModel(new SystemErrorException($this->response, $exception));
        }
    }

    public function delete($id)
    {
        try {
            /**
             * @var $activeAdmin Admin
             */
            $activeAdmin = $this->serviceLocator->get('ActiveAdmin');

            if (!$activeAdmin->enabled)
            {
                return new JsonModel(new AccessDeniedException($this->response, 'Administrative Access Required'));
            }

            /**
             * @var $seasonsTable SeasonsTable
             */
            $seasonsTable = $this->serviceLocator->get('Application/Model/SeasonsTable');

            $seasonsTable->deleteOne($id);

            return new JsonModel(new DeletedEntryNotificationException($this->response, "Season Deleted"));

        } catch (EntryNotFoundException $notFoundException)
        {
            return new \API\Exception\EntryNotFoundException($this->response,
                "UNABLE TO FIND ENTRY");
        } catch (\Exception $exception)
        {
            return new JsonModel(new SystemErrorException($this->response, $exception));
        }

    }
}
