<?php

namespace API\Controller;

use API\Exception\AccessDeniedException;
use API\Exception\DeletedEntryNotificationException;
use API\Exception\SystemErrorException;
use Application\Exception\EntryNotFoundException;
use Application\Model\Admin;
use Application\Model\ReservedDate;
use Application\Model\ReservedDatesTable;
use Laminas\Db\ResultSet\ResultSet;
use Laminas\Mvc\Controller\AbstractRestfulController;
use Laminas\View\Model\JsonModel;

class ReservedDatesController extends AbstractRestfulController
{
    public function get($id)
    {
        try
        {
            /**
             * @var $reservedDatesTable ReservedDatesTable
             */
            $reservedDatesTable = $this->serviceLocator->get('Application\Model\ReservedDatesTable');

            /**
             * @var $reservedDate ReservedDate
             */
            $reservedDate = $reservedDatesTable->fetchOne($id);

            return new JsonModel($reservedDate->toArray());

        } catch (EntryNotFoundException $notFoundException)
        {
            return new \API\Exception\EntryNotFoundException($this->response,
                "UNABLE TO FIND ENTRY");
        } catch (\Exception $exception)
        {
            return new JsonModel(new SystemErrorException($this->response, $exception));
        }
    }

    public function getList()
    {
        try
        {
            /**
             * @var $reservedDatesTable ReservedDatesTable
             */
            $reservedDatesTable = $this->serviceLocator->get('Application\Model\ReservedDatesTable');

            $startDate = $this->request->getQuery('startDate', 'now');
            $endDate = $this->request->getQuery('endDate', 'now');

            $startDate = strtotime($startDate);
            $endDate = strtotime($endDate);

            /**
             * @var $resultSet ResultSet
             */
            $resultSet = $reservedDatesTable->fetchCurrent($startDate, $endDate);

            $return = [];

            /**
             * @var $reservedDate ReservedDate
             */
            while ($reservedDate = $resultSet->current())
            {
                $return[] = $reservedDate->toArray();
                $resultSet->next();
            }

            return new JsonModel($return);

        } catch (EntryNotFoundException $notFoundException)
        {
            return new \API\Exception\EntryNotFoundException($this->response,
                "UNABLE TO FIND ENTRY");
        } catch (\Exception $exception)
        {
            return new JsonModel(new SystemErrorException($this->response, $exception));
        }
    }

    public function create($data)
    {
        try
        {
            /**
             * @var $activeAdmin Admin
             */
            $activeAdmin = $this->serviceLocator->get('ActiveAdmin');

            if (!$activeAdmin->enabled)
            {
                return new JsonModel(new AccessDeniedException($this->response, 'Administrative Access Required'));
            }

            /**
             * @var $reservedDatesTable ReservedDatesTable
             */
            $reservedDatesTable = $this->serviceLocator->get('Application\Model\ReservedDatesTable');

            $reservedDate = new ReservedDate();
            $reservedDate->exchangeArray($data);
            $reservedDate->emaildate = null;
            $reservedDate->id = null;

            $reservedDatesTable->saveOne($reservedDate);

            return new JsonModel($reservedDate->toArray());

        } catch (\Exception $exception)
        {
            return new JsonModel(new SystemErrorException($this->response, $exception));
        }
    }

    public function update($id, $data)
    {
        try
        {

            /**
             * @var $activeAdmin Admin
             */
            $activeAdmin = $this->serviceLocator->get('ActiveAdmin');

            if (!$activeAdmin->enabled)
            {
                return new JsonModel(new AccessDeniedException($this->response, 'Administrative Access Required'));
            }

            /**
             * @var $reservedDatesTable ReservedDatesTable
             */
            $reservedDatesTable = $this->serviceLocator->get('Application\Model\ReservedDatesTable');


            $reservedDate = new ReservedDate();
            $reservedDate->exchangeArray($data);

            if ($reservedDate->emaildate)
            {
                $reservedDate->emaildate = date("Y-m-d H:i:s", strtotime(preg_replace('/\(.*\)/', '', $reservedDate->emaildate)));
            } else
            {
                $reservedDate->emaildate = null;
            }

            $reservedDatesTable->saveOne($reservedDate);

            return new JsonModel($reservedDate->toArray());

        } catch (EntryNotFoundException $notFoundException)
        {
            return new \API\Exception\EntryNotFoundException($this->response,
                "UNABLE TO FIND ENTRY");
        } catch (\Exception $exception)
        {
            return new JsonModel(new SystemErrorException($this->response, $exception));
        }
    }

    public function delete($id)
    {
        try {
            /**
             * @var $activeAdmin Admin
             */
            $activeAdmin = $this->serviceLocator->get('ActiveAdmin');

            if (!$activeAdmin->enabled)
            {
                return new JsonModel(new AccessDeniedException($this->response, 'Administrative Access Required'));
            }

            /**
             * @var $reservedDatesTable ReservedDatesTable
             */
            $reservedDatesTable = $this->serviceLocator->get('Application\Model\ReservedDatesTable');

            $reservedDatesTable->deleteOne($id);

            return new JsonModel(new DeletedEntryNotificationException($this->response, "Reserved Date Removed"));

        } catch (EntryNotFoundException $notFoundException)
        {
            return new \API\Exception\EntryNotFoundException($this->response,
                "UNABLE TO FIND ENTRY");
        } catch (\Exception $exception)
        {
            return new JsonModel(new SystemErrorException($this->response, $exception));
        }

    }
}
